<?php
// add_user.php — Güvenli kullanıcı ekleme (UTF8MB4, InnoDB)
// Gerekenler: db.php, users tablosu (id, name, email, password_hash, is_active)
declare(strict_types=1);

$cookieParams = [
  'lifetime'=>0,'path'=>'/','domain'=>'',
  'secure'=> isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off',
  'httponly'=>true,'samesite'=>'Strict'
];
session_set_cookie_params($cookieParams);
session_start();

require_once __DIR__.'/db.php';

// CSRF
if (empty($_SESSION['csrf_token'])) {
  $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$flash = $_SESSION['flash'] ?? null; unset($_SESSION['flash']);

function flash($type,$msg){ $_SESSION['flash']=['type'=>$type,'msg'=>$msg]; }
function back(){ header('Location: add_user.php'); exit; }

// POST geldiyse ekle
if ($_SERVER['REQUEST_METHOD']==='POST') {
  if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], (string)$_POST['csrf_token'])) {
    flash('danger','Güvenlik doğrulaması başarısız.'); back();
  }

  $name  = trim((string)($_POST['name']  ?? ''));
  $email = trim((string)($_POST['email'] ?? ''));
  $pass  = (string)($_POST['password'] ?? '');
  $active= isset($_POST['is_active']) ? 1 : 0;

  // Basit doğrulamalar
  if ($name === '' || mb_strlen($name) < 2) { flash('warning','Ad soyad en az 2 karakter olmalı.'); back(); }
  if (!filter_var($email, FILTER_VALIDATE_EMAIL)) { flash('warning','Geçerli bir e-posta girin.'); back(); }
  if (strlen($pass) < 8) { flash('warning','Şifre en az 8 karakter olmalı.'); back(); }

  // Şifre karması
  $hash = password_hash($pass, PASSWORD_DEFAULT);

  // Eklemeyi dene
  try {
    $stmt = $pdo->prepare("INSERT INTO users (name,email,password_hash,is_active) VALUES (:n,:e,:h,:a)");
    $stmt->execute([':n'=>$name, ':e'=>$email, ':h'=>$hash, ':a'=>$active]);
    flash('success','Kullanıcı eklendi.'); back();
  } catch (PDOException $ex) {
    // 1062 = duplicate key (unique email)
    if ($ex->errorInfo[1] === 1062) {
      flash('danger','Bu e-posta zaten kayıtlı.');
    } else {
      // üretimde detay göstermeyin
      flash('danger','Beklenmeyen bir hata oluştu.');
    }
    back();
  }
}
?>
<!doctype html>
<html lang="tr">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Kullanıcı Ekle</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body{background:linear-gradient(180deg,#fff,#f7fafc);}
    .card{border-radius:16px;box-shadow:0 10px 30px rgba(2,8,23,.06)}
  </style>
</head>
<body class="py-5">
  <div class="container">
    <div class="row justify-content-center">
      <div class="col-12 col-lg-6">
        <div class="card p-4 p-md-5">
          <h1 class="h4 mb-3">Kullanıcı Ekle</h1>
          <p class="text-secondary mb-4">Yeni bir kullanıcıyı güvenli şekilde ekleyin.</p>

          <?php if($flash): ?>
            <div class="alert alert-<?= htmlspecialchars($flash['type']) ?>"><?= htmlspecialchars($flash['msg']) ?></div>
          <?php endif; ?>

          <form method="post" class="needs-validation" novalidate>
            <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($_SESSION['csrf_token']) ?>">
            <div class="mb-3">
              <label class="form-label">Ad Soyad</label>
              <input name="name" type="text" class="form-control" required minlength="2" placeholder="Örn: Ali Veli">
              <div class="invalid-feedback">Lütfen ad soyad girin.</div>
            </div>
            <div class="mb-3">
              <label class="form-label">E-posta</label>
              <input name="email" type="email" class="form-control" required placeholder="ornek@site.com" autocomplete="email">
              <div class="invalid-feedback">Geçerli bir e-posta girin.</div>
            </div>
            <div class="mb-3">
              <label class="form-label">Şifre</label>
              <input name="password" type="password" class="form-control" required minlength="8" placeholder="En az 8 karakter" autocomplete="new-password">
              <div class="invalid-feedback">Şifre en az 8 karakter olmalı.</div>
            </div>
            <div class="form-check mb-3">
              <input class="form-check-input" type="checkbox" id="is_active" name="is_active" checked>
              <label class="form-check-label" for="is_active">Hesap aktif</label>
            </div>
            <div class="d-grid">
              <button class="btn btn-primary" type="submit">Kaydet</button>
            </div>
          </form>

        </div>
        <p class="text-center small text-secondary mt-3">Not: E-posta tekildir. Şifreler `password_hash()` ile saklanır.</p>
      </div>
    </div>
  </div>
  <script>
  (()=>{const f=document.querySelector('.needs-validation'); if(!f) return;
    f.addEventListener('submit',e=>{ if(!f.checkValidity()){e.preventDefault();e.stopPropagation();} f.classList.add('was-validated'); });
  })();
  </script>
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js" defer></script>
</body>
</html>
