<?php
// dashboard.php — Müşteri listesi + müşteri ekle + müşteri sil, 5 alt sayfa linkli
declare(strict_types=1);
session_start();
require_once __DIR__ . '/db.php';

// Giriş kontrolü
if (empty($_SESSION['user'])) {
  header('Location: login.php'); exit;
}

// CSRF
if (empty($_SESSION['csrf_token'])) {
  $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf = $_SESSION['csrf_token'];

$flash = $_SESSION['flash'] ?? null; unset($_SESSION['flash']);
function flash($type,$msg){ $_SESSION['flash']=['type'=>$type,'msg'=>$msg]; }

// -------------------- MÜŞTERİ SİLME (POST) --------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'delete_customer') {
  if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], (string)$_POST['csrf_token'])) {
    flash('danger','Güvenlik doğrulaması başarısız.'); header('Location: dashboard.php'); exit;
  }
  $id = (int)($_POST['id'] ?? 0);
  if ($id <= 0) { flash('warning','Geçersiz müşteri.'); header('Location: dashboard.php'); exit; }

  // Silmeden önce kayıt var mı kontrol edelim (opsiyonel)
  $chk = $pdo->prepare("SELECT id, name FROM customers WHERE id = :id LIMIT 1");
  $chk->execute([':id'=>$id]);
  $cust = $chk->fetch();

  if (!$cust) {
    flash('warning','Müşteri bulunamadı.'); header('Location: dashboard.php'); exit;
  }

  try {
    $pdo->beginTransaction();
    // ON DELETE CASCADE varsa bağlı sales/receipts otomatik silinir
    $del = $pdo->prepare("DELETE FROM customers WHERE id = :id LIMIT 1");
    $del->execute([':id'=>$id]);
    $pdo->commit();
    flash('success','Müşteri silindi: '.$cust['name']);
  } catch (Throwable $e) {
    $pdo->rollBack();
    // Üretimde detay göstermeyin
    flash('danger','Müşteri silinirken hata oluştu.');
  }

  header('Location: dashboard.php'); exit;
}

// -------------------- MÜŞTERİ EKLEME (POST) --------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'add_customer') {
  if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], (string)$_POST['csrf_token'])) {
    flash('danger','Güvenlik doğrulaması başarısız.'); header('Location: dashboard.php'); exit;
  }

  $name    = trim((string)($_POST['name'] ?? ''));
  $email   = trim((string)($_POST['email'] ?? ''));
  $phone   = trim((string)($_POST['phone'] ?? ''));
  $company = trim((string)($_POST['company'] ?? ''));
  $note    = trim((string)($_POST['note'] ?? ''));

  // Basit doğrulamalar
  if ($name === '' || mb_strlen($name) < 2) {
    flash('warning','Müşteri adı en az 2 karakter olmalı.'); header('Location: dashboard.php'); exit;
  }
  if ($email !== '' && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
    flash('warning','Geçerli bir e-posta girin veya boş bırakın.'); header('Location: dashboard.php'); exit;
  }

  // Ekle
  $stmt = $pdo->prepare("INSERT INTO customers (name,email,phone,company,note) VALUES (:n,:e,:p,:c,:no)");
  $stmt->execute([
    ':n'=>$name, ':e'=>$email !== '' ? $email : null, ':p'=>$phone !== '' ? $phone : null,
    ':c'=>$company !== '' ? $company : null, ':no'=>$note !== '' ? $note : null
  ]);
  flash('success','Müşteri eklendi.'); header('Location: dashboard.php'); exit;
}

// -------------------- LİSTELEME (arama + sayfalama) --------------------
$q = trim((string)($_GET['q'] ?? ''));
$page = max(1, (int)($_GET['page'] ?? 1));
$perPage = 10;
$offset = ($page - 1) * $perPage;

$params = [];
$where  = '';
if ($q !== '') {
  $where = "WHERE name LIKE :q OR email LIKE :q OR phone LIKE :q OR company LIKE :q";
  $params[':q'] = "%{$q}%";
}

$total = (function($pdo,$where,$params){
  $stmt = $pdo->prepare("SELECT COUNT(*) FROM customers {$where}");
  $stmt->execute($params);
  return (int)$stmt->fetchColumn();
})($pdo,$where,$params);

$stmt = $pdo->prepare("
  SELECT id,name,email,phone,company,note,created_at
  FROM customers
  {$where}
  ORDER BY created_at DESC
  LIMIT :lim OFFSET :off
");
foreach ($params as $k=>$v) $stmt->bindValue($k,$v);
$stmt->bindValue(':lim',$perPage,PDO::PARAM_INT);
$stmt->bindValue(':off',$offset,PDO::PARAM_INT);
$stmt->execute();
$rows = $stmt->fetchAll();

$pages = max(1, (int)ceil($total / $perPage));
$user = $_SESSION['user'];
?>
<!doctype html>
<html lang="tr" data-bs-theme="light">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Dashboard</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
  <style>
    :root{
      --brand:#0ea5e9;
      --bg1:#f8fafc;
      --card: #ffffff;
    }
    body{background:linear-gradient(180deg,#fff 0%, var(--bg1) 100%);}
    .sidebar{min-height:100vh;border-right:1px solid rgba(0,0,0,.06);background:#fff;}
    .sidebar .nav-link{border-radius:10px;color:#334155;font-weight:500}
    .sidebar .nav-link.active, .sidebar .nav-link:hover{background:rgba(14,165,233,.1); color:#0ea5e9;}
    .card{border-radius:16px; box-shadow:0 10px 30px rgba(2,8,23,.06); border:1px solid rgba(0,0,0,.06);}
    .btn-brand{background:var(--brand);border:none}
    .badge-soft{background:rgba(14,165,233,.12); color:#0ea5e9;}
    .table thead th{white-space:nowrap}
    .brand-dot{width:10px;height:10px;border-radius:50%;background:var(--brand); display:inline-block;margin-right:.5rem;}
  </style>
</head>
<body>
<nav class="navbar navbar-expand-lg bg-white border-bottom">
  <div class="container-fluid">
    <a class="navbar-brand" href="#">
      <span class="brand-dot"></span>MRT Cari
    </a>
    <div class="d-flex align-items-center gap-3 ms-auto">
      <span class="text-secondary small d-none d-md-inline">Merhaba, <strong><?= htmlspecialchars($user['name']) ?></strong></span>
      <a class="btn btn-outline-secondary btn-sm" href="logout.php"><i class="bi bi-box-arrow-right me-1"></i> Çıkış</a>
    </div>
  </div>
</nav>

<div class="container-fluid">
  <div class="row">
    <!-- Sidebar -->
    <aside class="col-12 col-md-3 col-lg-2 p-3 sidebar">
      <div class="list-group list-group-flush">
        <div class="mb-2 small text-uppercase text-secondary">Menü</div>
        <a class="nav-link active mb-1" href="dashboard.php"><i class="bi bi-speedometer2 me-2"></i>Anasayfa</a>
        <!-- 5 Alt Sayfa (şimdilik yer tutucu bağlantılar) -->
        <a class="nav-link mb-1" href="page1.php"><i class="bi bi-collection me-2"></i>Kardeşler Müşteri</a>
        <a class="nav-link mb-1" href="page2.php"><i class="bi bi-receipt me-2"></i>Kardeşler Terzi</a>
        <a class="nav-link mb-1" href="page3.php"><i class="bi bi-diagram-3 me-2"></i>Miray Çarşı</a>
        <a class="nav-link mb-1" href="page4.php"><i class="bi bi-graph-up me-2"></i>Günlük Ciro - Pos Ciro</a>
      

        <div class="mb-2 small text-uppercase text-secondary">Müşteriler</div>
        <a class="nav-link mb-1" href="#musteri-liste"><i class="bi bi-people me-2"></i>Müşteri Listesi</a>
        <a class="nav-link" href="#musteri-ekle"><i class="bi bi-person-plus me-2"></i>Müşteri Ekle</a>
      </div>
    </aside>

    <!-- Content -->
    <main class="col-12 col-md-9 col-lg-10 p-3 p-md-4">
      <?php if ($flash): ?>
        <div class="alert alert-<?= htmlspecialchars($flash['type']) ?>"><?= htmlspecialchars($flash['msg']) ?></div>
      <?php endif; ?>

      <div class="row g-3 mb-4">
        <div class="col-12 col-lg-8">
          <div class="card p-3 p-md-4">
            <div class="d-flex align-items-center justify-content-between flex-wrap gap-2 mb-3">
              <h5 class="mb-0" id="musteri-liste">Müşteri Listesi</h5>
              <form class="d-flex" role="search" method="get">
                <input class="form-control form-control-sm me-2" type="search" name="q" value="<?= htmlspecialchars($q) ?>" placeholder="Ara: ad, e-posta, telefon..." aria-label="Ara">
                <button class="btn btn-outline-secondary btn-sm" type="submit"><i class="bi bi-search"></i></button>
              </form>
            </div>

            <div class="table-responsive">
              <table class="table table-hover align-middle mb-0">
                <thead>
                <tr>
                  <th>#</th>
                  <th>Ad</th>
                  <th>E-posta</th>
                  <th>Telefon</th>
                  <th>Firma</th>
                  <th>Not</th>
                  <th>Oluşturma</th>
                  <th style="width:90px" class="text-end">İşlem</th>
                </tr>
                </thead>
                <tbody>
                <?php if (!$rows): ?>
                  <tr><td colspan="8" class="text-center text-secondary py-4">Kayıt bulunamadı.</td></tr>
                <?php else: foreach ($rows as $r): ?>
                  <tr>
                    <td><?= (int)$r['id'] ?></td>
                    <td class="fw-600"><?= htmlspecialchars($r['name']) ?></td>
                    <td><?= htmlspecialchars($r['email'] ?? '') ?></td>
                    <td><?= htmlspecialchars($r['phone'] ?? '') ?></td>
                    <td><span class="badge rounded-pill badge-soft"><?= htmlspecialchars($r['company'] ?? '') ?></span></td>
                    <td class="text-truncate" style="max-width:240px"><?= htmlspecialchars($r['note'] ?? '') ?></td>
                    <td><small class="text-secondary"><?= htmlspecialchars($r['created_at']) ?></small></td>
                    <td class="text-end">
                      <form method="post" class="d-inline delete-form" data-name="<?= htmlspecialchars($r['name']) ?>">
                        <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrf) ?>">
                        <input type="hidden" name="action" value="delete_customer">
                        <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                        <button type="submit" class="btn btn-sm btn-outline-danger">
                          <i class="bi bi-trash me-1"></i> Sil
                        </button>
                      </form>
                    </td>
                  </tr>
                <?php endforeach; endif; ?>
                </tbody>
              </table>
            </div>

            <!-- Sayfalama -->
            <nav class="mt-3">
              <ul class="pagination pagination-sm mb-0">
                <li class="page-item <?= $page<=1?'disabled':''; ?>">
                  <a class="page-link" href="?<?= http_build_query(['q'=>$q,'page'=>max(1,$page-1)]) ?>">Önceki</a>
                </li>
                <?php
                  // kısa sayfalama (en fazla 7 buton)
                  $win = 3;
                  $start = max(1, $page-$win);
                  $end   = min($pages, $page+$win);
                  for ($i=$start;$i<=$end;$i++):
                ?>
                  <li class="page-item <?= $i===$page?'active':''; ?>">
                    <a class="page-link" href="?<?= http_build_query(['q'=>$q,'page'=>$i]) ?>"><?= $i ?></a>
                  </li>
                <?php endfor; ?>
                <li class="page-item <?= $page>=$pages?'disabled':''; ?>">
                  <a class="page-link" href="?<?= http_build_query(['q'=>$q,'page'=>min($pages,$page+1)]) ?>">Sonraki</a>
                </li>
              </ul>
            </nav>
          </div>
        </div>

        <div class="col-12 col-lg-4" id="musteri-ekle">
          <div class="card p-3 p-md-4">
            <h5 class="mb-3">Müşteri Ekle</h5>
            <form method="post" class="needs-validation" novalidate>
              <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($csrf) ?>">
              <input type="hidden" name="action" value="add_customer">
              <div class="mb-2">
                <label class="form-label">Ad <span class="text-danger">*</span></label>
                <input name="name" type="text" class="form-control" required minlength="2" placeholder="Örn: Ahmet Yılmaz">
                <div class="invalid-feedback">Ad en az 2 karakter olmalı.</div>
              </div>
              <div class="mb-2">
                <label class="form-label">E-posta</label>
                <input name="email" type="email" class="form-control" placeholder="ornek@site.com">
                <div class="form-text">Boş bırakılabilir.</div>
              </div>
              <div class="mb-2">
                <label class="form-label">Telefon</label>
                <input name="phone" type="text" class="form-control" placeholder="05xx xxx xx xx">
              </div>
              <div class="mb-2">
                <label class="form-label">Firma</label>
                <input name="company" type="text" class="form-control" placeholder="Firma adı">
              </div>
              <div class="mb-3">
                <label class="form-label">Not</label>
                <textarea name="note" rows="3" class="form-control" placeholder="Kısa not..."></textarea>
              </div>
              <div class="d-grid">
                <button class="btn btn-brand text-white" type="submit"><i class="bi bi-plus-circle me-1"></i> Ekle</button>
              </div>
            </form>
          </div>
        </div>
      </div>

      <!-- Hızlı kartlar -->
      <div class="row g-3">
        <div class="col-12 col-md-4">
          <div class="card p-3">
            <div class="d-flex justify-content-between align-items-center">
              <div>
                <div class="text-secondary small">Toplam Müşteri</div>
                <div class="h4 mb-0"><?= number_format($total,0,',','.') ?></div>
              </div>
              <i class="bi bi-people fs-1 text-secondary"></i>
            </div>
          </div>
        </div>
        <div class="col-12 col-md-4">
          <div class="card p-3">
            <div class="text-secondary small mb-1">Arama</div>
            <div class="small"><?= $q!=='' ? htmlspecialchars($q) : '—' ?></div>
          </div>
        </div>
      </div>

    </main>
  </div>
</div>

<script>
(() => {
  'use strict';
  // Form client-side validation
  const forms = document.querySelectorAll('.needs-validation');
  Array.from(forms).forEach(form => {
    form.addEventListener('submit', event => {
      if (!form.checkValidity()) { event.preventDefault(); event.stopPropagation(); }
      form.classList.add('was-validated');
    }, false);
  });

  // Silme onayı (event delegation)
  document.addEventListener('submit', function(e){
    const f = e.target.closest('.delete-form');
    if (f) {
      const name = f.getAttribute('data-name') || 'bu müşteri';
      const ok = confirm(`Müşteri "${name}" silinsin mi?\nBu işlem geri alınamaz.`);
      if (!ok) { e.preventDefault(); e.stopPropagation(); }
    }
  });
})();
</script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js" defer></script>
</body>
</html>
