<?php
// login.php — Güvenli kullanıcı girişi
declare(strict_types=1);

// Güvenli cookie parametreleri (PHP >= 7.3)
$cookieParams = [
    'lifetime' => 0,
    'path'     => '/',
    'domain'   => '',
    'secure'   => isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off',
    'httponly' => true,
    'samesite' => 'Strict'
];
session_set_cookie_params($cookieParams);
session_start();

require_once __DIR__ . '/db.php';

function client_ip(): string {
    // Proxy kullanmıyorsanız REMOTE_ADDR yeterli; proxy arkasında güvenli bir şekilde X-Forwarded-For doğrulayın.
    return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
}

function ip_to_binary(string $ip): string {
    $packed = @inet_pton($ip);
    return $packed !== false ? $packed : inet_pton('0.0.0.0');
}

// CSRF token üretimi
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Basit mesaj tutucu
$flash = $_SESSION['flash'] ?? null;
unset($_SESSION['flash']);

// Rate limit ayarları
$MAX_FAILS    = 5;                 // izin verilen başarısız deneme
$WINDOW_MIN   = 15;                // dakika
$BLOCK_MSG    = "Çok sayıda başarısız deneme tespit edildi. Lütfen {$WINDOW_MIN} dakika sonra tekrar deneyin.";

// POST işle
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // 1) CSRF kontrol
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], (string)$_POST['csrf_token'])) {
        $_SESSION['flash'] = ['type' => 'danger', 'msg' => 'Güvenlik doğrulaması başarısız. Lütfen tekrar deneyin.'];
        header('Location: login.php'); exit;
    }

    // 2) Rate limit kontrol (IP veya e-posta bazlı)
    $email = trim((string)($_POST['email'] ?? ''));
    $ipBin = ip_to_binary(client_ip());

    $stmt = $pdo->prepare("
        SELECT COUNT(*) AS fail_count
        FROM login_attempts
        WHERE success = 0
          AND (ip_address = :ip OR email = :email)
          AND created_at >= (NOW() - INTERVAL :window MINUTE)
    ");
    $stmt->bindValue(':ip', $ipBin, PDO::PARAM_LOB);
    $stmt->bindValue(':email', $email);
    $stmt->bindValue(':window', $WINDOW_MIN, PDO::PARAM_INT);
    $stmt->execute();
    $fails = (int)$stmt->fetchColumn();

    if ($fails >= $MAX_FAILS) {
        $_SESSION['flash'] = ['type' => 'warning', 'msg' => $BLOCK_MSG];
        header('Location: login.php'); exit;
    }

    // 3) Girdi doğrulama (temel)
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $_SESSION['flash'] = ['type' => 'danger', 'msg' => 'Geçerli bir e-posta giriniz.'];
        header('Location: login.php'); exit;
    }

    $password = (string)($_POST['password'] ?? '');

    // 4) Kullanıcı sorgusu
    $stmt = $pdo->prepare("SELECT id, name, email, password_hash, is_active FROM users WHERE email = :email LIMIT 1");
    $stmt->execute([':email' => $email]);
    $user = $stmt->fetch();

    $ok = false;
    if ($user && (int)$user['is_active'] === 1) {
        $ok = password_verify($password, $user['password_hash']);
        // opsiyonel: password_needs_rehash ile rehash edebilirsiniz
    }

    // 5) Deneme kaydı
    $stmt = $pdo->prepare("INSERT INTO login_attempts (email, ip_address, success) VALUES (:email, :ip, :success)");
    $stmt->bindValue(':email', $email);
    $stmt->bindValue(':ip', $ipBin, PDO::PARAM_LOB);
    $stmt->bindValue(':success', $ok ? 1 : 0, PDO::PARAM_INT);
    $stmt->execute();

    if ($ok) {
        // 6) Oturum başlat
        session_regenerate_id(true);
        $_SESSION['user'] = [
            'id'    => (int)$user['id'],
            'name'  => $user['name'],
            'email' => $user['email'],
        ];
        // Basit yönlendirme
        header('Location: dashboard.php'); exit;
    } else {
        $_SESSION['flash'] = ['type' => 'danger', 'msg' => 'E-posta veya şifre hatalı.'];
        header('Location: login.php'); exit;
    }
}
?>
<!doctype html>
<html lang="tr" data-bs-theme="light">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Giriş Yap | Uygulama</title>
  <meta name="color-scheme" content="light only">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
  <style>
    :root{
      --brand:#0ea5e9; /* cyan-500 */
      --bg-grad-1:#0ea5e933;
      --bg-grad-2:#22d3ee33;
    }
    body{
      min-height:100vh;
      background:
        radial-gradient(60% 60% at 100% 0%, var(--bg-grad-2), transparent 60%),
        radial-gradient(60% 60% at 0% 100%, var(--bg-grad-1), transparent 60%),
        linear-gradient(180deg,#ffffff 0%, #f7fafc 100%);
    }
    .auth-card{
      backdrop-filter: blur(10px);
      border:1px solid rgba(0,0,0,.06);
      box-shadow: 0 10px 30px rgba(2, 8, 23, .06);
      border-radius: 18px;
    }
    .brand-dot{
      width:12px;height:12px;border-radius:50%;background:var(--brand);
      display:inline-block;margin-right:.5rem;
      box-shadow:0 0 0 6px rgba(14,165,233,.15);
    }
    .form-control{
      padding:.8rem 1rem;border-radius:12px;
    }
    .btn-brand{
      background:var(--brand);border:none;border-radius:12px;
    }
    .btn-brand:hover{filter:brightness(.95);}
  </style>
</head>
<body>
  <div class="container py-5">
    <div class="row justify-content-center align-items-center">
      <div class="col-12 col-md-10 col-lg-8 col-xl-5">
        <div class="p-4 p-md-5 bg-white auth-card">
          <div class="text-center mb-4">
            <div class="mb-2">
              <span class="brand-dot"></span>
              <span class="fw-bold">NetCode</span>
            </div>
            <h1 class="h4 fw-700 mb-1">Hesabınıza Giriş Yapın</h1>
            <p class="text-secondary mb-0">Devam etmek için e-posta ve şifrenizi girin.</p>
          </div>

          <?php if ($flash): ?>
            <div class="alert alert-<?= htmlspecialchars($flash['type']) ?>"><?= htmlspecialchars($flash['msg']) ?></div>
          <?php endif; ?>

          <form method="post" novalidate class="needs-validation">
            <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($_SESSION['csrf_token']) ?>">
            <div class="mb-3">
              <label for="email" class="form-label">E-posta</label>
              <input type="email" class="form-control" id="email" name="email" required placeholder="ornek@site.com" autocomplete="username">
              <div class="invalid-feedback">Lütfen geçerli bir e-posta girin.</div>
            </div>
            <div class="mb-2">
              <label for="password" class="form-label">Şifre</label>
              <div class="input-group">
                <input type="password" class="form-control" id="password" name="password" required minlength="8" autocomplete="current-password" placeholder="••••••••">
                <button class="btn btn-outline-secondary" type="button" onclick="togglePass()"><i class="bi bi-eye"></i></button>
              </div>
              <div class="invalid-feedback">Şifrenizi girin.</div>
            </div>
            <div class="d-grid mt-4">
              <button class="btn btn-brand btn-lg text-white" type="submit">
                <i class="bi bi-box-arrow-in-right me-2"></i> Giriş Yap
              </button>
            </div>
          </form>

          <div class="text-center mt-4 small text-secondary">
            <span>© <?= date('Y') ?> NetCode — Tüm hakları saklıdır.</span>
          </div>
        </div>
        <p class="text-center small text-secondary mt-3">Güvenli bağlantı: <i class="bi bi-shield-lock"></i> HTTPS etkin olmalı.</p>
      </div>
    </div>
  </div>

  <script>
    // Bootstrap client-side validation
    (() => {
      'use strict';
      const forms = document.querySelectorAll('.needs-validation');
      Array.from(forms).forEach(form => {
        form.addEventListener('submit', event => {
          if (!form.checkValidity()) {
            event.preventDefault(); event.stopPropagation();
          }
          form.classList.add('was-validated');
        }, false);
      });
    })();

    function togglePass(){
      const i = document.getElementById('password');
      i.type = (i.type === 'password') ? 'text' : 'password';
    }
  </script>
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js" defer></script>
</body>
</html>
