<?php
// page1.php — "Excel benzeri" cari sayfası (Satışlar + Tahsilatlar + Bakiye + BORÇLAR listesi)
declare(strict_types=1);
session_start();
require_once __DIR__ . '/db.php';

if (empty($_SESSION['user'])) { header('Location: login.php'); exit; }

// CSRF
if (empty($_SESSION['csrf_token'])) { $_SESSION['csrf_token'] = bin2hex(random_bytes(32)); }
$csrf = $_SESSION['csrf_token'];

// Müşteriler
$customers = $pdo->query("SELECT id,name FROM customers ORDER BY name ASC")->fetchAll();
if (!$customers) { die('Önce müşteri ekleyin.'); }

// Aktif müşteri
$cid = isset($_GET['cid']) ? (int)$_GET['cid'] : (int)$customers[0]['id'];
$activeCustomer = null; foreach ($customers as $c){ if ((int)$c['id'] === $cid) { $activeCustomer = $c; break; } }
if (!$activeCustomer) { http_response_code(404); die('Müşteri bulunamadı'); }

// SAVE endpoint (JSON)
if ($_SERVER['REQUEST_METHOD']==='POST' && ($_GET['action'] ?? '')==='save') {
    if (!isset($_SERVER['HTTP_X_CSRF_TOKEN']) || !hash_equals($csrf, (string)$_SERVER['HTTP_X_CSRF_TOKEN'])) {
        http_response_code(403); echo json_encode(['ok'=>false,'msg'=>'CSRF']); exit;
    }
    $raw = file_get_contents('php://input');
    $data = json_decode($raw, true);
    if (!is_array($data)) { http_response_code(400); echo json_encode(['ok'=>false,'msg'=>'Bad JSON']); exit; }

    $sales = $data['sales'] ?? [];
    $receipts = $data['receipts'] ?? [];

    $safeSales = [];
    foreach ($sales as $row) {
        $d = $row['date'] ?? '';
        $item = trim((string)($row['item'] ?? ''));
        $qty = (float)($row['qty'] ?? 0);
        $price = (float)($row['price'] ?? 0);
        $amount = round($qty * $price, 2);
        if ($d && $item !== '' && $qty > 0) {
            $safeSales[] = [$cid, $d, $item, $qty, $price, $amount];
        }
    }
    $safeReceipts = [];
    foreach ($receipts as $row) {
        $d = $row['date'] ?? '';
        $desc = trim((string)($row['desc'] ?? ''));
        $amt = (float)($row['amount'] ?? 0);
        if ($d && $desc !== '' && $amt > 0) {
            $safeReceipts[] = [$cid, $d, $desc, $amt];
        }
    }

    $pdo->beginTransaction();
    try {
        $st = $pdo->prepare("DELETE FROM sales WHERE customer_id = ?");
        $st->execute([$cid]);
        $st = $pdo->prepare("DELETE FROM receipts WHERE customer_id = ?");
        $st->execute([$cid]);

        if ($safeSales) {
            $st = $pdo->prepare("INSERT INTO sales (customer_id,s_date,item,qty,price,amount) VALUES (?,?,?,?,?,?)");
            foreach ($safeSales as $r) { $st->execute($r); }
        }
        if ($safeReceipts) {
            $st = $pdo->prepare("INSERT INTO receipts (customer_id,r_date,description,amount) VALUES (?,?,?,?)");
            foreach ($safeReceipts as $r) { $st->execute($r); }
        }

        $pdo->commit();
        echo json_encode(['ok'=>true]);
    } catch (Throwable $e) {
        $pdo->rollBack();
        http_response_code(500);
        echo json_encode(['ok'=>false,'msg'=>'DB error']);
    }
    exit;
}

// Aktif müşterinin verileri
$rowsSales = $pdo->prepare("SELECT s_date AS date, item, qty, price, amount FROM sales WHERE customer_id=? ORDER BY s_date ASC, id ASC");
$rowsSales->execute([$cid]);
$salesData = $rowsSales->fetchAll();

$rowsReceipts = $pdo->prepare("SELECT r_date AS date, description AS `desc`, amount FROM receipts WHERE customer_id=? ORDER BY r_date ASC, id ASC");
$rowsReceipts->execute([$cid]);
$receiptsData = $rowsReceipts->fetchAll();

// BORÇLAR paneli için tüm bakiyeler
$debtsStmt = $pdo->query("
  SELECT 
    c.id,
    c.name,
    COALESCE(s.total_sales, 0)   AS sales_total,
    COALESCE(r.total_receipts, 0) AS receipts_total,
    COALESCE(s.total_sales, 0) - COALESCE(r.total_receipts, 0) AS balance
  FROM customers c
  LEFT JOIN (
    SELECT customer_id, SUM(amount) AS total_sales
    FROM sales GROUP BY customer_id
  ) s ON s.customer_id = c.id
  LEFT JOIN (
    SELECT customer_id, SUM(amount) AS total_receipts
    FROM receipts GROUP BY customer_id
  ) r ON r.customer_id = c.id
  ORDER BY c.name ASC
");
$debtsData = $debtsStmt->fetchAll();
$debtsTotal = 0.0;
foreach ($debtsData as $d) { $debtsTotal += (float)$d['balance']; }

$user = $_SESSION['user'];
?>
<!doctype html>
<html lang="tr" data-bs-theme="light">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>Cari Sayfası</title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
<style>
  body{background:#f7fafc;}
  .sheet-card{border-radius:16px;border:1px solid rgba(0,0,0,.06);box-shadow:0 10px 30px rgba(2,8,23,.06);}
  .zone-title{font-weight:700;letter-spacing:.3px}
  .grid th, .grid td{vertical-align:middle}
  .grid thead th{background:#e5e7eb; position:sticky; top:0; z-index:2}
  .grid input{border:0;background:transparent; width:100%; padding:.35rem .5rem; outline:none}
  .grid input:focus{background:#fffbe6; box-shadow: inset 0 0 0 1px #f59e0b; border-radius:8px}
  .grid .row-add{cursor:pointer}
  .section-head{background:#e5e7eb;border-radius:10px;padding:.3rem .6rem;display:inline-block}
  .badge-soft{background:rgba(14,165,233,.12);color:#0ea5e9}
  .totals{font-weight:700}
  .sticky-tabs{position:sticky; bottom:0; background:#ffffff;border-top:1px solid rgba(0,0,0,.06); padding:.4rem}

  /* BORÇLAR paneli */
  #debtsPanel .pos-neg { font-weight:600; }
  #debtsPanel .pos-neg.positive { color:#0ea5e9; }
  #debtsPanel .pos-neg.negative { color:#ef4444; }
</style>
</head>
<body>
<nav class="navbar navbar-light bg-white border-bottom mb-3">
  <div class="container-fluid">
    <a class="navbar-brand" href="dashboard.php"><i class="bi bi-arrow-left-circle me-2"></i>Anasayfa</a>
    <div class="d-flex align-items-center gap-3">
      <span class="small text-secondary">Kullanıcı: <strong><?= htmlspecialchars($user['name']) ?></strong></span>
    </div>
  </div>
</nav>

<div class="container">
  <div class="d-flex align-items-center justify-content-between mb-3">
    <h4 class="mb-0"><span class="badge badge-soft rounded-pill me-2">&nbsp;</span><?= htmlspecialchars($activeCustomer['name']) ?> - Cari</h4>
    <div class="d-flex gap-2">
      <button id="btnAddSale" class="btn btn-outline-secondary btn-sm"><i class="bi bi-plus-square me-1"></i>Satır (Satış)</button>
      <button id="btnAddRec" class="btn btn-outline-secondary btn-sm"><i class="bi bi-plus-square-dotted me-1"></i>Satır (Tahsilat)</button>
      <button id="btnSave" class="btn btn-primary btn-sm"><i class="bi bi-save2 me-1"></i>Kaydet</button>
    </div>
  </div>

  <!-- DETAY KISMI: Satış + Tahsilat + Bakiye  -->
  <div id="detailWrap">
    <div class="row g-3 align-items-start">
      <!-- Sol: Satış + Tahsilat -->
      <div class="col-12 col-lg-9">
        <div class="sheet-card p-3 p-md-4 mb-3">
          <div class="zone-title mb-2"><span class="section-head">Satışlar</span></div>
          <div class="table-responsive">
            <table class="table grid align-middle mb-2" id="tblSales">
              <thead>
                <tr>
                  <th style="width:140px">Tarih</th>
                  <th>Malzeme Adı</th>
                  <th style="width:120px">Adet</th>
                  <th style="width:140px">Fiyat</th>
                  <th style="width:140px">Tutar</th>
                  <th style="width:48px"></th>
                </tr>
              </thead>
              <tbody></tbody>
              <tfoot>
                <tr>
                  <td colspan="4" class="text-end fw-600">TOPLAM</td>
                  <td class="totals" id="salesTotal">0</td>
                  <td></td>
                </tr>
              </tfoot>
            </table>
          </div>
        </div>

        <div class="sheet-card p-3 p-md-4">
          <div class="zone-title mb-2"><span class="section-head">Tahsilatlar</span></div>
          <div class="table-responsive">
            <table class="table grid align-middle mb-2" id="tblReceipts">
              <thead>
                <tr>
                  <th style="width:140px">Tarih</th>
                  <th>Açıklama</th>
                  <th style="width:140px">Tutar</th>
                  <th style="width:48px"></th>
                </tr>
              </thead>
              <tbody></tbody>
              <tfoot>
                <tr>
                  <td colspan="2" class="text-end fw-600">TOPLAM</td>
                  <td class="totals" id="recTotal">0</td>
                  <td></td>
                </tr>
              </tfoot>
            </table>
          </div>
        </div>
      </div>

      <!-- Sağ: Bakiye -->
      <div class="col-12 col-lg-3">
        <div class="sheet-card p-3 p-md-4">
          <div class="zone-title mb-2"><span class="section-head">Bakiye</span></div>
          <div class="display-6" id="balance">0</div>
          <div class="small text-secondary">Bakiye = Satışlar Toplamı − Tahsilatlar Toplamı</div>
        </div>
      </div>
    </div>
  </div>

  <!-- BORÇLAR Paneli (başlangıçta gizli) -->
  <div id="debtsPanel" class="sheet-card p-3 p-md-4 mt-3 d-none">
    <div class="d-flex align-items-center justify-content-between mb-2">
      <div class="zone-title"><span class="section-head">BORÇLAR</span></div>
      <small class="text-secondary">Tüm müşterilerin bakiye listesi</small>
    </div>
    <div class="table-responsive">
      <table class="table align-middle mb-2">
        <thead class="table-light">
          <tr>
            <th style="width:56px">#</th>
            <th>Müşteri</th>
            <th class="text-end">Satışlar</th>
            <th class="text-end">Tahsilatlar</th>
            <th class="text-end">Bakiye</th>
            <th style="width:140px" class="text-end">Git</th>
          </tr>
        </thead>
        <tbody>
          <?php
          $i=1;
          foreach ($debtsData as $d):
            $sales  = (float)$d['sales_total'];
            $recs   = (float)$d['receipts_total'];
            $bal    = (float)$d['balance'];
            $cls    = $bal >= 0 ? 'positive' : 'negative';
          ?>
            <tr>
              <td><?= $i++ ?></td>
              <td><?= htmlspecialchars($d['name']) ?></td>
              <td class="text-end"><?= number_format($sales, 2, ',', '.') ?></td>
              <td class="text-end"><?= number_format($recs, 2, ',', '.') ?></td>
              <td class="text-end pos-neg <?= $cls ?>"><?= number_format($bal, 2, ',', '.') ?></td>
              <td class="text-end">
                <a class="btn btn-sm btn-outline-primary" href="?cid=<?= (int)$d['id'] ?>">
                  <i class="bi bi-box-arrow-in-right me-1"></i> Aç
                </a>
              </td>
            </tr>
          <?php endforeach; ?>
        </tbody>
        <tfoot>
          <tr>
            <th colspan="4" class="text-end">GENEL TOPLAM</th>
            <th class="text-end"><?= number_format($debtsTotal, 2, ',', '.') ?></th>
            <th></th>
          </tr>
        </tfoot>
      </table>
    </div>
  </div>

  <!-- Müşteri Sekmeleri + BORÇLAR sekmesi -->
  <div class="sticky-tabs mt-3">
    <ul class="nav nav-pills flex-wrap gap-1 align-items-center">
      <?php foreach ($customers as $c): ?>
        <li class="nav-item">
          <a class="nav-link customer-tab <?= ((int)$c['id']===$cid)?'active':''; ?>" href="?cid=<?= (int)$c['id'] ?>">
            <?= htmlspecialchars($c['name']) ?>
          </a>
        </li>
      <?php endforeach; ?>
      <li class="nav-item ms-2">
        <a id="tabDebts" class="nav-link btn-light" href="#"><i class="bi bi-journal-text me-1"></i>BORÇLAR</a>
      </li>
    </ul>
  </div>
</div>

<script>
const csrfToken = "<?= htmlspecialchars($csrf) ?>";
const cid = <?= (int)$cid ?>;

// Başlangıç verileri
const initialSales = <?= json_encode($salesData, JSON_UNESCAPED_UNICODE) ?>;
const initialReceipts = <?= json_encode($receiptsData, JSON_UNESCAPED_UNICODE) ?>;

// ---------- Sayı format/parsing ----------
const nf = new Intl.NumberFormat('tr-TR', {minimumFractionDigits:2, maximumFractionDigits:2});
const fmt = n => nf.format(Number.isFinite(+n) ? +n : 0);
function toNum(v){
  if (v == null) return 0;
  v = String(v).trim().replace(/\s/g,'');
  if (v === '') return 0;
  if (v.includes(',') && v.includes('.')) { v = v.replace(/\./g,'').replace(',', '.'); }
  else if (v.includes(',')) { v = v.replace(',', '.'); }
  v = v.replace(/[^0-9.+-]/g,'');
  const n = parseFloat(v);
  return Number.isFinite(n) ? n : 0;
}

// ---------- Satır ekleme ----------
function addSaleRow(row={date:'', item:'', qty:'', price:'', amount:''}){
  const tr = document.createElement('tr');
  tr.innerHTML = `
    <td><input type="date" value="${row.date||''}"></td>
    <td><input type="text" placeholder="Malzeme" value="${row.item||''}"></td>
    <td><input type="text" inputmode="decimal" value="${row.qty||''}"></td>
    <td><input type="text" inputmode="decimal" value="${row.price||''}"></td>
    <td><input type="text" value="${row.amount||''}" readonly></td>
    <td class="text-center"><button class="btn btn-sm btn-outline-danger row-del"><i class="bi bi-x-lg"></i></button></td>
  `;
  document.querySelector('#tblSales tbody').appendChild(tr);
}
function addRecRow(row={date:'', desc:'', amount:''}){
  const tr = document.createElement('tr');
  tr.innerHTML = `
    <td><input type="date" value="${row.date||''}"></td>
    <td><input type="text" placeholder="Açıklama" value="${row.desc||''}"></td>
    <td><input type="text" inputmode="decimal" value="${row.amount||''}"></td>
    <td class="text-center"><button class="btn btn-sm btn-outline-danger row-del"><i class="bi bi-x-lg"></i></button></td>
  `;
  document.querySelector('#tblReceipts tbody').appendChild(tr);
}

// ---------- Hesaplamalar ----------
function recalc(){
  let sTot = 0;
  document.querySelectorAll('#tblSales tbody tr').forEach(tr=>{
    const qty   = toNum(tr.children[2].querySelector('input').value);
    const price = toNum(tr.children[3].querySelector('input').value);
    const amt   = Math.round((qty*price + Number.EPSILON) * 100) / 100;
    tr.children[4].querySelector('input').value = fmt(amt);
    sTot += amt;
  });
  document.getElementById('salesTotal').textContent = fmt(sTot);

  let rTot = 0;
  document.querySelectorAll('#tblReceipts tbody tr').forEach(tr=>{
    const amt = toNum(tr.children[2].querySelector('input').value);
    rTot += amt;
  });
  document.getElementById('recTotal').textContent = fmt(rTot);

  document.getElementById('balance').textContent = fmt(sTot - rTot);
}

function collectData(){
  const sales = [];
  document.querySelectorAll('#tblSales tbody tr').forEach(tr=>{
    const d = tr.children[0].querySelector('input').value;
    const item = tr.children[1].querySelector('input').value.trim();
    const qty   = toNum(tr.children[2].querySelector('input').value);
    const price = toNum(tr.children[3].querySelector('input').value);
    if(d && item && qty>0){ sales.push({date:d, item, qty, price}); }
  });
  const receipts = [];
  document.querySelectorAll('#tblReceipts tbody tr').forEach(tr=>{
    const d = tr.children[0].querySelector('input').value;
    const desc = tr.children[1].querySelector('input').value.trim();
    const amount = toNum(tr.children[2].querySelector('input').value);
    if(d && desc && amount>0){ receipts.push({date:d, desc, amount}); }
  });
  return {sales, receipts};
}

async function saveAll(){
  const payload = collectData();
  const res = await fetch(`page1.php?action=save&cid=${cid}`, {
    method:'POST',
    headers: {'Content-Type':'application/json','X-CSRF-Token': csrfToken},
    body: JSON.stringify(payload)
  });
  if(!res.ok){ alert('Kaydetme hatası'); return; }
  const j = await res.json();
  if(j.ok){ alert('Kaydedildi'); } else { alert('Hata: '+(j.msg||'')); }
}

// ---------- Eventler ----------
document.getElementById('btnAddSale').addEventListener('click', ()=>{ addSaleRow(); });
document.getElementById('btnAddRec').addEventListener('click', ()=>{ addRecRow(); });
document.getElementById('btnSave').addEventListener('click', saveAll);

document.addEventListener('input', e=>{
  if(e.target.closest('#tblSales') || e.target.closest('#tblReceipts')) recalc();
});

// Silme onayı
document.addEventListener('click', e=>{
  const delBtn = e.target.closest('.row-del');
  if(delBtn){
    e.preventDefault();
    if (confirm('Satırı silmek istiyor musunuz?')) {
      const tr = delBtn.closest('tr');
      tr.parentNode.removeChild(tr);
      recalc();
    }
  }
});

// ---------- BORÇLAR sekmesi: sadece borç listesi görünsün ----------
const debtsPanel = document.getElementById('debtsPanel');
const detailWrap = document.getElementById('detailWrap');
const tabDebts   = document.getElementById('tabDebts');

function showDebtsOnly(){
  debtsPanel.classList.remove('d-none');   // borçlar açık
  detailWrap.classList.add('d-none');      // detaylar gizli
  tabDebts.classList.add('active');        // sekme aktif

  // müşteri sekmelerindeki 'active' sınıfını temizle (görsel netlik)
  document.querySelectorAll('.customer-tab').forEach(a=>a.classList.remove('active'));
}
function hideDebts(){
  debtsPanel.classList.add('d-none');
  detailWrap.classList.remove('d-none');
  tabDebts.classList.remove('active');
}

tabDebts?.addEventListener('click', (e)=>{ e.preventDefault(); 
  if (debtsPanel.classList.contains('d-none')) showDebtsOnly(); else hideDebts();
});

// Müşteri sekmesine geçince detaylar geri gelsin
document.querySelectorAll('.customer-tab').forEach(a=>{
  a.addEventListener('click', ()=>{ hideDebts(); });
});

// ---------- Init ----------
(initialSales.length?initialSales:[{date:'',item:'',qty:'',price:'',amount:''}]).forEach(addSaleRow);
(initialReceipts.length?initialReceipts:[{date:'',desc:'',amount:''}]).forEach(addRecRow);
recalc();
</script>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js" defer></script>
</body>
</html>
