<?php
// page4.php — Ciro Yönetimi (Günlük Ciro, POS Ciro, Toplam Ciro)
declare(strict_types=1);
session_start();
require_once __DIR__ . '/db.php';

if (empty($_SESSION['user'])) { header('Location: login.php'); exit; }

if (empty($_SESSION['csrf_token'])) { $_SESSION['csrf_token'] = bin2hex(random_bytes(32)); }
$csrf = $_SESSION['csrf_token'];
$user = $_SESSION['user'];

// ---------- JSON Yardımcıları ----------
function json_out($arr, int $code=200){ http_response_code($code); header('Content-Type: application/json; charset=utf-8'); echo json_encode($arr, JSON_UNESCAPED_UNICODE); exit; }

// ---------- SAVE endpointleri ----------
// action=save&type=daily  (payload: [{date,amount},...])
// action=save&type=pos
if ($_SERVER['REQUEST_METHOD']==='POST' && ($_GET['action'] ?? '')==='save') {
  if (empty($_SESSION['user'])) json_out(['ok'=>false,'msg'=>'AUTH'],401);
  if (!isset($_SERVER['HTTP_X_CSRF_TOKEN']) || !hash_equals($csrf,(string)$_SERVER['HTTP_X_CSRF_TOKEN'])) json_out(['ok'=>false,'msg'=>'CSRF'],403);

  $type = $_GET['type'] ?? '';
  if (!in_array($type,['daily','pos'],true)) json_out(['ok'=>false,'msg'=>'type'],400);

  $payload = json_decode(file_get_contents('php://input'), true);
  if (!is_array($payload)) json_out(['ok'=>false,'msg'=>'Bad JSON'],400);

  $rows = $payload['rows'] ?? [];
  $safe = [];
  foreach ($rows as $r){
    $d = $r['date'] ?? '';
    $amt = (float)($r['amount'] ?? 0);
    if ($d && $amt>0){ $safe[] = [$d, round($amt,2)]; }
  }

  $pdo->beginTransaction();
  try{
    if ($type==='daily'){
      $pdo->exec("DELETE FROM daily_revenue");
      if ($safe){
        $ins=$pdo->prepare("INSERT INTO daily_revenue (r_date,amount) VALUES (?,?)");
        foreach($safe as $x){ $ins->execute($x); }
      }
    } else {
      $pdo->exec("DELETE FROM pos_revenue");
      if ($safe){
        $ins=$pdo->prepare("INSERT INTO pos_revenue (r_date,amount) VALUES (?,?)");
        foreach($safe as $x){ $ins->execute($x); }
      }
    }
    $pdo->commit();
    json_out(['ok'=>true]);
  }catch(Throwable $e){
    $pdo->rollBack();
    json_out(['ok'=>false,'msg'=>'DB error'],500);
  }
}

// ---------- Aggregation endpoint ----------
// action=agg&year=YYYY&month=MM&from=YYYY-MM-DD&to=YYYY-MM-DD
if (($_GET['action'] ?? '')==='agg') {
  $year  = isset($_GET['year'])  && $_GET['year']!==''  ? (int)$_GET['year']  : null;
  $month = isset($_GET['month']) && $_GET['month']!=='' ? (int)$_GET['month'] : null;
  $from  = $_GET['from'] ?? null;
  $to    = $_GET['to']   ?? null;

  // Esnek filtre WHERE’leri
  $wDaily = $wPos = [];
  $pDaily = $pPos = [];

  if ($year){ $wDaily[]="YEAR(r_date)=?"; $pDaily[]=$year; $wPos[]="YEAR(r_date)=?"; $pPos[]=$year; }
  if ($month){ $wDaily[]="MONTH(r_date)=?"; $pDaily[]=$month; $wPos[]="MONTH(r_date)=?"; $pPos[]=$month; }
  if ($from){ $wDaily[]="r_date>=?"; $pDaily[]=$from; $wPos[]="r_date>=?"; $pPos[]=$from; }
  if ($to){   $wDaily[]="r_date<=?"; $pDaily[]=$to;   $wPos[]="r_date<=?";   $pPos[]=$to; }

  $wd = $wDaily ? ('WHERE '.implode(' AND ',$wDaily)) : '';
  $wp = $wPos   ? ('WHERE '.implode(' AND ',$wPos))   : '';

  // Aylık bazda toplamları çek (YYYY-MM)
  $sqlD = "SELECT DATE_FORMAT(r_date,'%Y-%m') ym, SUM(amount) s FROM daily_revenue $wd GROUP BY ym ORDER BY ym";
  $sqlP = "SELECT DATE_FORMAT(r_date,'%Y-%m') ym, SUM(amount) s FROM pos_revenue   $wp GROUP BY ym ORDER BY ym";

  $dMap = []; $pMap = []; $rows = [];

  $st=$pdo->prepare($sqlD); $st->execute($pDaily); foreach($st->fetchAll() as $r){ $dMap[$r['ym']]=(float)$r['s']; }
  $st=$pdo->prepare($sqlP); $st->execute($pPos);   foreach($st->fetchAll() as $r){ $pMap[$r['ym']]=(float)$r['s']; }

  $months = array_values(array_unique(array_merge(array_keys($dMap), array_keys($pMap))));
  sort($months);
  $grandDaily=$grandPos=0.0;

  foreach($months as $m){
    $d = $dMap[$m] ?? 0.0; $p = $pMap[$m] ?? 0.0;
    $rows[] = ['month'=>$m, 'daily'=>$d, 'pos'=>$p, 'total'=>$d+$p];
    $grandDaily += $d; $grandPos += $p;
  }

  $yearsStmt = $pdo->query("
      SELECT DISTINCT YEAR(r_date) y FROM daily_revenue
      UNION
      SELECT DISTINCT YEAR(r_date) y FROM pos_revenue
      ORDER BY y DESC
  ");
  $years = array_map(fn($x)=>(int)$x['y'], $yearsStmt->fetchAll());

  json_out([
    'ok'=>true,
    'rows'=>$rows,
    'sum'=>['daily'=>$grandDaily, 'pos'=>$grandPos, 'total'=>$grandDaily+$grandPos],
    'years'=>$years
  ]);
}

// ---------- Sayfaya ilk verileri yükle ----------
$daily = $pdo->query("SELECT r_date AS date, amount FROM daily_revenue ORDER BY r_date ASC")->fetchAll();
$pos   = $pdo->query("SELECT r_date AS date, amount FROM pos_revenue   ORDER BY r_date ASC")->fetchAll();
?>
<!doctype html>
<html lang="tr" data-bs-theme="light">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>Ciro Yönetimi</title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
<style>
  body{background:#f7fafc;}
  .cardx{border-radius:16px;border:1px solid rgba(0,0,0,.06);box-shadow:0 10px 30px rgba(2,8,23,.06);}
  .grid th,.grid td{vertical-align:middle}
  .grid thead th{background:#e5e7eb;position:sticky;top:0;z-index:2}
  .grid input{border:0;background:transparent;width:100%;padding:.35rem .5rem;outline:none}
  .grid input:focus{background:#fffbe6;box-shadow:inset 0 0 0 1px #f59e0b;border-radius:8px}
  .totals{font-weight:700}
</style>
</head>
<body>
<nav class="navbar navbar-light bg-white border-bottom mb-3">
  <div class="container-fluid">
    <a class="navbar-brand" href="dashboard.php"><i class="bi bi-arrow-left-circle me-2"></i>Anasayfa</a>
    <span class="small text-secondary">Kullanıcı: <strong><?= htmlspecialchars($user['name']) ?></strong></span>
  </div>
</nav>

<div class="container">
  <ul class="nav nav-pills mb-3" id="pills">
    <li class="nav-item"><button class="nav-link active" data-bs-toggle="pill" data-bs-target="#tabDaily">Günlük Ciro</button></li>
    <li class="nav-item"><button class="nav-link" data-bs-toggle="pill" data-bs-target="#tabPos">POS Ciro</button></li>
    <li class="nav-item"><button class="nav-link" data-bs-toggle="pill" data-bs-target="#tabTotal">Toplam Ciro</button></li>
  </ul>

  <div class="tab-content">
    <!-- Günlük Ciro -->
    <div class="tab-pane fade show active" id="tabDaily">
      <div class="cardx p-3 p-md-4">
        <div class="d-flex justify-content-between align-items-center mb-2">
          <h5 class="mb-0">Günlük Ciro</h5>
          <button id="btnSaveDaily" class="btn btn-primary btn-sm"><i class="bi bi-save2 me-1"></i>Kaydet</button>
        </div>
        <div class="table-responsive">
          <table class="table grid align-middle mb-2" id="tblDaily">
            <thead>
              <tr><th style="width:160px">Tarih</th><th style="width:200px">Tutar</th><th style="width:48px"></th></tr>
            </thead>
            <tbody></tbody>
            <tfoot>
              <tr><td class="text-end fw-600">TOPLAM</td><td class="totals" id="dailyTotal">0</td><td></td></tr>
            </tfoot>
          </table>
        </div>
        <button id="btnAddDaily" class="btn btn-outline-secondary btn-sm"><i class="bi bi-plus-square me-1"></i>Satır</button>
      </div>
    </div>

    <!-- POS Ciro -->
    <div class="tab-pane fade" id="tabPos">
      <div class="cardx p-3 p-md-4">
        <div class="d-flex justify-content-between align-items-center mb-2">
          <h5 class="mb-0">POS Ciro</h5>
          <button id="btnSavePos" class="btn btn-primary btn-sm"><i class="bi bi-save2 me-1"></i>Kaydet</button>
        </div>
        <div class="table-responsive">
          <table class="table grid align-middle mb-2" id="tblPos">
            <thead>
              <tr><th style="width:160px">Tarih</th><th style="width:200px">Tutar</th><th style="width:48px"></th></tr>
            </thead>
            <tbody></tbody>
            <tfoot>
              <tr><td class="text-end fw-600">TOPLAM</td><td class="totals" id="posTotal">0</td><td></td></tr>
            </tfoot>
          </table>
        </div>
        <button id="btnAddPos" class="btn btn-outline-secondary btn-sm"><i class="bi bi-plus-square me-1"></i>Satır</button>
      </div>
    </div>

    <!-- Toplam Ciro (Filtreler + Özet) -->
    <div class="tab-pane fade" id="tabTotal">
      <div class="cardx p-3 p-md-4 mb-3">
        <div class="d-flex flex-wrap gap-2 align-items-end">
          <div>
            <label class="form-label mb-1">Yıl</label>
            <select id="filterYear" class="form-select form-select-sm" style="min-width:120px">
              <option value="">Tümü</option>
            </select>
          </div>
          <div>
            <label class="form-label mb-1">Ay</label>
            <select id="filterMonth" class="form-select form-select-sm" style="min-width:120px">
              <option value="">Tümü</option>
              <?php for($m=1;$m<=12;$m++): ?>
                <option value="<?= $m ?>"><?= $m ?></option>
              <?php endfor; ?>
            </select>
          </div>
          <div>
            <label class="form-label mb-1">Başlangıç</label>
            <input type="date" id="filterFrom" class="form-control form-control-sm">
          </div>
          <div>
            <label class="form-label mb-1">Bitiş</label>
            <input type="date" id="filterTo" class="form-control form-control-sm">
          </div>
          <div>
            <button id="btnFilter" class="btn btn-outline-secondary btn-sm"><i class="bi bi-funnel me-1"></i>Filtrele</button>
          </div>
        </div>
      </div>

      <div class="row g-3 mb-3">
        <div class="col-12 col-md-4">
          <div class="cardx p-3 text-center">
            <div class="text-secondary small">Günlük Ciro Toplamı</div>
            <div class="h4 mb-0" id="sumDaily">0</div>
          </div>
        </div>
        <div class="col-12 col-md-4">
          <div class="cardx p-3 text-center">
            <div class="text-secondary small">POS Ciro Toplamı</div>
            <div class="h4 mb-0" id="sumPos">0</div>
          </div>
        </div>
        <div class="col-12 col-md-4">
          <div class="cardx p-3 text-center">
            <div class="text-secondary small">GENEL TOPLAM</div>
            <div class="h4 mb-0" id="sumTotal">0</div>
          </div>
        </div>
      </div>

      <div class="cardx p-3 p-md-4">
        <div class="table-responsive">
          <table class="table align-middle">
            <thead class="table-light">
              <tr>
                <th>Ay</th>
                <th class="text-end">Günlük Ciro</th>
                <th class="text-end">POS Ciro</th>
                <th class="text-end">Toplam</th>
              </tr>
            </thead>
            <tbody id="aggBody"></tbody>
          </table>
        </div>
      </div>
    </div>
  </div>
</div>

<!-- Toast -->
<div class="position-fixed top-0 end-0 p-3" style="z-index:1080">
  <div id="appToast" class="toast align-items-center text-bg-success border-0" role="alert" aria-live="assertive" aria-atomic="true">
    <div class="d-flex">
      <div class="toast-body" id="appToastBody">Kaydedildi</div>
      <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Kapat"></button>
    </div>
  </div>
</div>

<script>
const csrfToken = "<?= htmlspecialchars($csrf) ?>";
const initialDaily = <?= json_encode($daily, JSON_UNESCAPED_UNICODE) ?>;
const initialPos   = <?= json_encode($pos,   JSON_UNESCAPED_UNICODE) ?>;

const nf = new Intl.NumberFormat('tr-TR',{minimumFractionDigits:2,maximumFractionDigits:2});
const fmt = n => nf.format(Number.isFinite(+n)?+n:0);
function toNum(v){ if(v==null) return 0; v=String(v).trim().replace(/\s/g,''); if(v==='') return 0;
  if(v.includes(',')&&v.includes('.')) v=v.replace(/\./g,'').replace(',', '.'); else if(v.includes(',')) v=v.replace(',', '.');
  v=v.replace(/[^0-9.+-]/g,''); const n=parseFloat(v); return Number.isFinite(n)?n:0; }

function notify(msg, type='success'){
  const t=document.getElementById('appToast'); const b=document.getElementById('appToastBody');
  b.textContent=msg; t.className='toast align-items-center border-0 text-bg-'+type;
  new bootstrap.Toast(t,{delay:1800}).show();
}

// ---- Satır ekle
function addRow(tbodyId, row={date:'',amount:''}){
  const tr=document.createElement('tr');
  tr.innerHTML = `
    <td><input type="date" value="${row.date||''}"></td>
    <td><input type="text" inputmode="decimal" value="${row.amount||''}"></td>
    <td class="text-center"><button class="btn btn-sm btn-outline-danger row-del"><i class="bi bi-x-lg"></i></button></td>`;
  document.querySelector(`#${tbodyId}`).appendChild(tr);
}
function totalize(tbodyId, outId){
  let tot=0; document.querySelectorAll(`#${tbodyId} tr`).forEach(tr=>{
    tot += toNum(tr.children[1].querySelector('input').value);
  });
  document.getElementById(outId).textContent = fmt(tot);
  return tot;
}
function collect(tbodyId){
  const rows=[]; document.querySelectorAll(`#${tbodyId} tr`).forEach(tr=>{
    const d=tr.children[0].querySelector('input').value;
    const a=toNum(tr.children[1].querySelector('input').value);
    if(d && a>0) rows.push({date:d, amount:a});
  }); return rows;
}
async function saveGrid(type, tbodyId){
  const payload={rows:collect(tbodyId)};
  const res=await fetch(`page4.php?action=save&type=${type}`,{
    method:'POST',
    headers:{'Content-Type':'application/json','X-CSRF-Token':csrfToken},
    credentials:'same-origin',
    body:JSON.stringify(payload)
  });
  const j=await res.json();
  if(res.ok && j.ok){ notify('Kaydedildi','success'); } else { notify('Hata','danger'); }
}

// ---- Günlük
(initialDaily.length?initialDaily:[{date:'',amount:''}]).forEach(r=>addRow('tblDaily tbody',r));
document.getElementById('dailyTotal').textContent = fmt(totalize('tblDaily tbody','dailyTotal'));

// ---- POS
(initialPos.length?initialPos:[{date:'',amount:''}]).forEach(r=>addRow('tblPos tbody',r));
document.getElementById('posTotal').textContent = fmt(totalize('tblPos tbody','posTotal'));

// Input & delete handlers
document.addEventListener('input', e=>{
  if(e.target.closest('#tblDaily')) totalize('tblDaily tbody','dailyTotal');
  if(e.target.closest('#tblPos'))   totalize('tblPos tbody','posTotal');
});
document.addEventListener('click', e=>{
  const del=e.target.closest('.row-del'); if(del){ e.preventDefault(); del.closest('tr').remove();
    if(e.target.closest('#tblDaily')) totalize('tblDaily tbody','dailyTotal');
    if(e.target.closest('#tblPos'))   totalize('tblPos tbody','posTotal');
  }
});

// Buttons
document.getElementById('btnAddDaily').addEventListener('click', ()=>addRow('tblDaily tbody'));
document.getElementById('btnAddPos').addEventListener('click',   ()=>addRow('tblPos tbody'));
document.getElementById('btnSaveDaily').addEventListener('click', ()=>saveGrid('daily','tblDaily tbody'));
document.getElementById('btnSavePos').addEventListener('click',   ()=>saveGrid('pos','tblPos tbody'));

// ---- Toplam Ciro: filtre ve liste
async function loadAgg(){
  const y=document.getElementById('filterYear').value;
  const m=document.getElementById('filterMonth').value;
  const f=document.getElementById('filterFrom').value;
  const t=document.getElementById('filterTo').value;
  const params = new URLSearchParams({action:'agg'});
  if(y) params.set('year',y); if(m) params.set('month',m);
  if(f) params.set('from',f); if(t) params.set('to',t);
  const res = await fetch('page4.php?'+params.toString(), {credentials:'same-origin'});
  const j = await res.json();
  if(!j.ok) return;
  // yıl seçenekleri (ilk yüklemede doldur)
  const selY=document.getElementById('filterYear');
  if(selY.options.length<=1){
    j.years.forEach(yy=>{ const o=document.createElement('option'); o.value=yy; o.textContent=yy; selY.appendChild(o); });
  }
  // toplam rozetleri
  document.getElementById('sumDaily').textContent = fmt(j.sum.daily);
  document.getElementById('sumPos').textContent   = fmt(j.sum.pos);
  document.getElementById('sumTotal').textContent = fmt(j.sum.total);
  // tablo
  const tb=document.getElementById('aggBody'); tb.innerHTML='';
  j.rows.forEach(r=>{
    const tr=document.createElement('tr');
    tr.innerHTML = `
      <td>${r.month}</td>
      <td class="text-end">${fmt(r.daily)}</td>
      <td class="text-end">${fmt(r.pos)}</td>
      <td class="text-end fw-semibold">${fmt(r.total)}</td>`;
    tb.appendChild(tr);
  });
}
document.getElementById('btnFilter').addEventListener('click', loadAgg);

// “Toplam Ciro” sekmesi açılınca otomatik yükle
document.querySelector('[data-bs-target="#tabTotal"]').addEventListener('shown.bs.tab', loadAgg);
</script>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js" defer></script>
</body>
</html>
